const axios = require('axios');
const FormData = require('form-data');
const fs = require('fs');

/**
 * Upload file to FileVila CDN
 * @param {Object} file - Multer file object
 * @param {string} subfolder - Subfolder name (e.g., 'news', 'research', 'members', 'content')
 * @returns {Promise<string>} - FileVila CDN URL
 */
async function uploadToFileVila(file, subfolder = 'general') {
  try {
    // Verify file exists
    if (!file || !file.path) {
      throw new Error('Invalid file object - no path provided');
    }
    
    if (!fs.existsSync(file.path)) {
      throw new Error(`File not found at path: ${file.path}`);
    }

    console.log('Uploading to FileVila:', {
      filePath: file.path,
      subfolder,
      apiUrl: process.env.FILEVILA_API_URL
    });

    const formData = new FormData();
    formData.append('file', fs.createReadStream(file.path));
    formData.append('subfolder', subfolder);

    const uploadUrl = `${process.env.FILEVILA_API_URL}/api/files/upload`;
    console.log('Upload URL:', uploadUrl);

    const response = await axios.post(
      uploadUrl,
      formData,
      {
        headers: {
          ...formData.getHeaders(),
          'X-API-Key': process.env.FILEVILA_API_KEY
        },
        maxBodyLength: Infinity,
        maxContentLength: Infinity
      }
    );

    // Delete local file after successful upload
    try {
      fs.unlinkSync(file.path);
    } catch (err) {
      console.warn('Failed to delete local file:', err.message);
    }

    return response.data.file.url;
  } catch (error) {
    console.error('FileVila upload error details:');
    console.error('- Status:', error.response?.status);
    console.error('- Data:', error.response?.data);
    console.error('- Message:', error.message);
    console.error('- FILEVILA_API_URL:', process.env.FILEVILA_API_URL);
    console.error('- FILEVILA_API_KEY:', process.env.FILEVILA_API_KEY ? 'Set' : 'NOT SET');
    throw new Error(error.response?.data?.error || error.message || 'Failed to upload file to CDN');
  }
}

/**
 * Delete file from FileVila CDN
 * @param {string} fileId - FileVila file ID
 * @returns {Promise<void>}
 */
async function deleteFromFileVila(fileId) {
  try {
    await axios.delete(
      `${process.env.FILEVILA_API_URL}/api/files/${fileId}`,
      {
        headers: {
          'Authorization': `Bearer ${process.env.FILEVILA_ADMIN_TOKEN}`
        }
      }
    );
  } catch (error) {
    console.warn('FileVila delete warning:', error.response?.data || error.message);
    // Don't throw - deletion failure shouldn't block the main operation
  }
}

module.exports = {
  uploadToFileVila,
  deleteFromFileVila
};
