const express = require('express');
const router = express.Router();
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const { auth, staffOrAdmin } = require('../middleware/auth');
const { uploadToFileVila } = require('../utils/filevilaUpload');
const Media = require('../models/Media');

// Ensure upload directory exists
const uploadDir = path.join(__dirname, '../../uploads/temp');
if (!fs.existsSync(uploadDir)) {
  fs.mkdirSync(uploadDir, { recursive: true });
}

// Configure multer for media files
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, 'temp-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage: storage,
  limits: {
    fileSize: 50 * 1024 * 1024 // 50MB
  }
});

// Get all media files from database
router.get('/', auth, staffOrAdmin, async (req, res) => {
  try {
    const { folder, type, search, page = 1, limit = 100 } = req.query;
    
    // Build query
    let query = {};
    if (folder) query.folder = folder;
    if (type) query.type = new RegExp('^' + type, 'i'); // e.g., 'image' matches 'image/jpeg'
    if (search) {
      query.$or = [
        { originalName: new RegExp(search, 'i') },
        { filename: new RegExp(search, 'i') }
      ];
    }

    const skip = (page - 1) * limit;

    // Fetch media from database
    const media = await Media.find(query)
      .populate('uploadedBy', 'name email')
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit));
    
    const total = await Media.countDocuments(query);

    // Transform to match expected format
    const formattedMedia = media.map(item => ({
      _id: item._id,
      name: item.originalName,
      filename: item.filename,
      url: item.url,
      type: item.type,
      size: item.formattedSize || formatBytes(item.size),
      folder: item.folder,
      uploadedBy: item.uploadedBy,
      createdAt: item.createdAt
    }));
    
    res.json({ 
      media: formattedMedia,
      pagination: {
        total,
        page: parseInt(page),
        limit: parseInt(limit),
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching media:', error);
    res.status(500).json({ error: 'Failed to fetch media files' });
  }
});

// Upload media files
router.post('/upload', auth, staffOrAdmin, upload.array('files', 10), async (req, res) => {
  try {
    if (!req.files || req.files.length === 0) {
      return res.status(400).json({ error: 'No files uploaded' });
    }

    const { folder = 'media' } = req.body;

    // Upload to FileVila
    const uploadPromises = req.files.map(file => uploadToFileVila(file, folder));
    const uploadedUrls = await Promise.all(uploadPromises);

    // Save to database
    const mediaDocuments = await Promise.all(
      req.files.map(async (file, index) => {
        const mediaDoc = new Media({
          filename: file.filename,
          originalName: file.originalname,
          url: uploadedUrls[index],
          type: file.mimetype,
          size: file.size,
          formattedSize: formatBytes(file.size),
          folder: folder,
          uploadedBy: req.user._id
        });
        
        await mediaDoc.save();
        return mediaDoc;
      })
    );

    // Populate uploadedBy for response
    await Media.populate(mediaDocuments, { path: 'uploadedBy', select: 'name email' });

    // Format response
    const uploadedFiles = mediaDocuments.map(doc => ({
      _id: doc._id,
      name: doc.originalName,
      filename: doc.filename,
      url: doc.url,
      type: doc.type,
      size: doc.formattedSize,
      folder: doc.folder,
      uploadedBy: doc.uploadedBy,
      createdAt: doc.createdAt
    }));
    
    res.json({
      success: true,
      files: uploadedFiles
    });
  } catch (error) {
    console.error('Upload error:', error);
    res.status(500).json({ error: 'File upload failed: ' + error.message });
  }
});

// Get single media file by ID
router.get('/:id', auth, staffOrAdmin, async (req, res) => {
  try {
    const { id } = req.params;
    
    // Validate MongoDB ObjectId
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid media ID' });
    }

    const media = await Media.findById(id)
      .populate('uploadedBy', 'name email');
    
    if (!media) {
      return res.status(404).json({ error: 'Media file not found' });
    }

    res.json({
      media: {
        _id: media._id,
        name: media.originalName,
        filename: media.filename,
        url: media.url,
        type: media.type,
        size: media.formattedSize || formatBytes(media.size),
        folder: media.folder,
        uploadedBy: media.uploadedBy,
        createdAt: media.createdAt
      }
    });
  } catch (error) {
    console.error('Error fetching media:', error);
    res.status(500).json({ error: 'Failed to fetch media file' });
  }
});

// Delete media file from database
router.delete('/:id', auth, staffOrAdmin, async (req, res) => {
  try {
    const { id } = req.params;
    
    // Validate MongoDB ObjectId
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid media ID' });
    }

    const media = await Media.findById(id);
    
    if (!media) {
      return res.status(404).json({ error: 'Media file not found' });
    }

    // Delete from database
    await Media.findByIdAndDelete(id);

    // Note: FileVila files remain accessible via URL
    // If you want to delete from FileVila too, you'd need to implement that

    res.json({ success: true, message: 'Media file deleted successfully' });
  } catch (error) {
    console.error('Delete error:', error);
    res.status(500).json({ error: 'Failed to delete file: ' + error.message });
  }
});

// Helper function to format bytes
function formatBytes(bytes, decimals = 2) {
  if (bytes === 0) return '0 Bytes';
  const k = 1024;
  const dm = decimals < 0 ? 0 : decimals;
  const sizes = ['Bytes', 'KB', 'MB', 'GB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

module.exports = router;
