const mongoose = require('mongoose');

const mediaSchema = new mongoose.Schema({
  filename: {
    type: String,
    required: true,
    trim: true
  },
  originalName: {
    type: String,
    required: true,
    trim: true
  },
  url: {
    type: String,
    required: true
  },
  type: {
    type: String,
    required: true // e.g., 'image/jpeg', 'application/pdf'
  },
  size: {
    type: Number,
    required: true // in bytes
  },
  formattedSize: {
    type: String // e.g., '2.5 MB'
  },
  folder: {
    type: String,
    default: 'media' // Can be 'media', 'research', 'news', etc.
  },
  uploadedBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true
  }
}, {
  timestamps: true
});

// Index for faster queries
mediaSchema.index({ uploadedBy: 1, createdAt: -1 });
mediaSchema.index({ type: 1 });
mediaSchema.index({ folder: 1 });

// Helper method to format bytes
mediaSchema.methods.formatFileSize = function() {
  const bytes = this.size;
  if (bytes === 0) return '0 Bytes';
  const k = 1024;
  const sizes = ['Bytes', 'KB', 'MB', 'GB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
};

module.exports = mongoose.model('Media', mediaSchema);
