const rateLimit = require('express-rate-limit');
const slowDown = require('express-slow-down');
const helmet = require('helmet');
const { body, validationResult } = require('express-validator');

// Basic rate limiter for general API endpoints (Vercel optimized)
const basicRateLimit = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // Limit each IP to 100 requests per windowMs
  message: {
    error: 'Too many requests from this IP, please try again later.',
    retryAfter: '15 minutes'
  },
  standardHeaders: true, // Return rate limit info in headers
  legacyHeaders: false
  // Using default in-memory store (perfect for serverless functions)
});

// Strict rate limiter for authentication endpoints (Vercel optimized)
const authRateLimit = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 5, // Limit each IP to 5 login attempts per windowMs
  message: {
    error: 'Too many login attempts from this IP, please try again later.',
    retryAfter: '15 minutes'
  },
  standardHeaders: true,
  legacyHeaders: false,
  skipSuccessfulRequests: true // Don't count successful requests
  // Using default in-memory store (perfect for serverless)
});

// Mobile app rate limiter (more lenient)
const mobileRateLimit = rateLimit({
  windowMs: 1 * 60 * 1000, // 1 minute
  max: 30, // Limit each IP to 30 requests per minute
  message: {
    error: 'Too many requests from mobile app, please try again later.',
    retryAfter: '1 minute'
  },
  standardHeaders: true,
  legacyHeaders: false
});

// Speed limiter to slow down requests after threshold
const speedLimiter = slowDown({
  windowMs: 5 * 60 * 1000, // 5 minutes (admin-friendly)
  delayAfter: 50, // Allow first 50 requests per windowMs at full speed
  delayMs: () => 500, // Add 500ms delay per request after delayAfter (new v2 syntax)
  maxDelayMs: 3000, // Maximum delay is 3 seconds (admin-friendly)
  validate: { delayMs: false } // Disable the warning
});

// Helmet security configuration
const helmetConfig = helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'"],
      scriptSrc: ["'self'"],
      imgSrc: ["'self'", "data:", "https:"],
    },
  },
  crossOriginEmbedderPolicy: false, // Disable for API
  crossOriginResourcePolicy: { policy: "cross-origin" }
});

// Request size limiter
const requestSizeLimit = (limit = '10mb') => {
  return (req, res, next) => {
    const contentLength = req.get('Content-Length');
    if (contentLength && parseInt(contentLength) > parseFloat(limit) * 1024 * 1024) {
      return res.status(413).json({
        error: 'Request entity too large',
        maxSize: limit
      });
    }
    next();
  };
};

// IP whitelist middleware (for admin functions)
const ipWhitelist = (whitelist = []) => {
  return (req, res, next) => {
    if (whitelist.length === 0) {
      return next(); // No whitelist configured, allow all
    }

    const clientIP = req.ip || req.connection.remoteAddress || req.socket.remoteAddress;
    
    if (whitelist.includes(clientIP) || whitelist.includes('*')) {
      return next();
    }

    console.warn(`Blocked request from non-whitelisted IP: ${clientIP}`);
    return res.status(403).json({
      error: 'Access denied from this IP address'
    });
  };
};

// Advanced DDoS protection (Vercel serverless optimized)
const ddosProtection = async (req, res, next) => {
  const clientIP = req.ip;
  const now = Date.now();
  const window = 60 * 1000; // 1 minute window
  const maxRequests = 200; // Max requests per minute per IP
  
  try {
    // Use simple in-memory tracking (perfect for serverless functions)
    if (!global.ddosTracker) {
      global.ddosTracker = new Map();
    }
    
    const tracker = global.ddosTracker;
    const ipData = tracker.get(clientIP) || { count: 0, resetTime: now + window };
    
    if (now > ipData.resetTime) {
      ipData.count = 1;
      ipData.resetTime = now + window;
    } else {
      ipData.count++;
    }
    
    tracker.set(clientIP, ipData);
    
    if (ipData.count > maxRequests) {
      console.warn(`DDoS protection triggered for IP: ${clientIP} (${ipData.count} requests)`);
      return res.status(429).json({
        error: 'Rate limit exceeded. Possible DDoS attack detected.',
        retryAfter: '1 minute'
      });
    }
  } catch (error) {
    console.error('DDoS protection error:', error);
    // Continue on error to not block legitimate requests
  }
  
  next();
};

// Request validation middleware
const validateRequest = (validations) => {
  return async (req, res, next) => {
    // Run all validations
    await Promise.all(validations.map(validation => validation.run(req)));

    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        error: 'Validation failed',
        details: errors.array()
      });
    }

    next();
  };
};

// Common validation rules
const validationRules = {
  email: body('email').isEmail().normalizeEmail(),
  password: body('password').isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
  mongoId: (field) => body(field).isMongoId().withMessage(`${field} must be a valid MongoDB ID`),
  version: body('version').matches(/^\d+\.\d+\.\d+$/).withMessage('Version must be in format x.y.z')
};

module.exports = {
  basicRateLimit,
  authRateLimit,
  speedLimiter,
  helmetConfig,
  requestSizeLimit,
  ipWhitelist,
  ddosProtection,
  validateRequest,
  validationRules
};