const Research = require('../models/Research');
const { uploadToFileVila } = require('../utils/filevilaUpload');

// Get all research (with filters)
const getAllResearch = async (req, res) => {
  try {
    const { page = 1, limit = 10, status, tag, search = '' } = req.query;
    const skip = (page - 1) * limit;

    let query = {};
    
    // For public API, only show published
    if (status === 'published') {
      query.status = 'published';
    }
    
    // For admin, can filter by any status
    if (status && status !== 'published') {
      query.status = status;
    }

    if (tag) {
      query.tags = tag;
    }

    if (search) {
      query.$or = [
        { title: { $regex: search, $options: 'i' } },
        { short_desc: { $regex: search, $options: 'i' } },
        { tags: { $regex: search, $options: 'i' } }
      ];
    }

    const research = await Research.find(query)
      .populate('created_by', 'name email')
      .populate('tagged_members', 'name designation image')
      .sort({ published_at: -1, createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit));

    const total = await Research.countDocuments(query);

    res.json({
      research,
      pagination: {
        current_page: parseInt(page),
        total_pages: Math.ceil(total / limit),
        total_items: total,
        has_next: page * limit < total,
        has_prev: page > 1
      }
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get single research by ID
const getResearchById = async (req, res) => {
  try {
    const { id } = req.params;
    
    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid research ID format' });
    }
    
    const research = await Research.findById(id)
      .populate('created_by', 'name email')
      .populate('tagged_members', 'name designation image bio emails');

    if (!research) {
      return res.status(404).json({ error: 'Research not found' });
    }

    res.json({ research });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Create research
const createResearch = async (req, res) => {
  try {
    let { title, short_desc, content, tags, tagged_members, status } = req.body;

    // Parse JSON strings if they're sent from FormData
    if (typeof tags === 'string') {
      try {
        tags = JSON.parse(tags);
      } catch (e) {
        tags = [];
      }
    }
    
    if (typeof tagged_members === 'string') {
      try {
        tagged_members = JSON.parse(tagged_members);
      } catch (e) {
        tagged_members = [];
      }
    }
    
    if (typeof content === 'string') {
      try {
        content = JSON.parse(content);
      } catch (e) {
        // Content might already be an object or valid JSON string
      }
    }

    if (!title || !content) {
      return res.status(400).json({ error: 'Title and content are required' });
    }

    if (title.trim().length < 3) {
      return res.status(400).json({ error: 'Title must be at least 3 characters long' });
    }

    if (title.length > 200) {
      return res.status(400).json({ error: 'Title must not exceed 200 characters' });
    }

    let thumbImageUrl = '';
    if (req.file) {
      // New file upload
      thumbImageUrl = await uploadToFileVila(req.file, 'research');
    } else if (req.body.thumb_image) {
      // URL from media library
      thumbImageUrl = req.body.thumb_image;
    }

    const research = new Research({
      title,
      short_desc: short_desc || '',
      content,
      tags: tags || [],
      tagged_members: tagged_members || [],
      status: status || 'draft',
      created_by: req.user._id,
      thumb_image: thumbImageUrl
    });

    if (status === 'published') {
      research.published_at = new Date();
    }

    await research.save();

    const populatedResearch = await Research.findById(research._id)
      .populate('created_by', 'name email')
      .populate('tagged_members', 'name designation image');

    res.status(201).json({
      message: 'Research created successfully',
      research: populatedResearch
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update research
const updateResearch = async (req, res) => {
  try {
    const { id } = req.params;
    let { title, short_desc, content, tags, tagged_members, status } = req.body;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid research ID format' });
    }

    // Parse JSON strings if they're sent from FormData
    if (typeof tags === 'string') {
      try {
        tags = JSON.parse(tags);
      } catch (e) {
        tags = undefined;
      }
    }
    
    if (typeof tagged_members === 'string') {
      try {
        tagged_members = JSON.parse(tagged_members);
      } catch (e) {
        tagged_members = undefined;
      }
    }
    
    if (typeof content === 'string') {
      try {
        content = JSON.parse(content);
      } catch (e) {
        // Content might already be an object or valid JSON string
      }
    }

    const research = await Research.findById(id);
    if (!research) {
      return res.status(404).json({ error: 'Research not found' });
    }

    if (title) research.title = title;
    if (short_desc !== undefined) research.short_desc = short_desc;
    if (content) research.content = content;
    if (tags) research.tags = tags;
    if (tagged_members) research.tagged_members = tagged_members;
    
    if (req.file) {
      // New file upload
      research.thumb_image = await uploadToFileVila(req.file, 'research');
    } else if (req.body.thumb_image !== undefined) {
      // URL from media library or empty string to clear
      research.thumb_image = req.body.thumb_image;
    }

    // Handle status change to published
    if (status === 'published' && research.status !== 'published') {
      research.status = 'published';
      research.published_at = new Date();
    } else if (status) {
      research.status = status;
    }

    await research.save();

    const populatedResearch = await Research.findById(research._id)
      .populate('created_by', 'name email')
      .populate('tagged_members', 'name designation image');

    res.json({
      message: 'Research updated successfully',
      research: populatedResearch
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Delete research
const deleteResearch = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid research ID format' });
    }

    const research = await Research.findById(id);
    if (!research) {
      return res.status(404).json({ error: 'Research not found' });
    }

    await Research.findByIdAndDelete(id);

    res.json({ message: 'Research deleted successfully' });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Publish research
const publishResearch = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid research ID format' });
    }

    const research = await Research.findById(id);
    if (!research) {
      return res.status(404).json({ error: 'Research not found' });
    }

    await research.publish();

    res.json({
      message: 'Research published successfully',
      research
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  getAllResearch,
  getResearchById,
  createResearch,
  updateResearch,
  deleteResearch,
  publishResearch
};
