const Publication = require('../models/Publication');

// Get all publications (public)
exports.getAllPublications = async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 50,
      type,
      year,
      featured,
      tag,
      search,
      sortBy = 'year',
      order = 'desc'
    } = req.query;

    const query = { isPublished: true };

    // Filters
    if (type) query.type = type;
    if (year) query.year = parseInt(year);
    if (featured !== undefined) query.featured = featured === 'true';
    if (tag) query.tags = tag;
    
    // Search
    if (search) {
      query.$or = [
        { title: { $regex: search, $options: 'i' } },
        { authors: { $regex: search, $options: 'i' } },
        { abstract: { $regex: search, $options: 'i' } }
      ];
    }

    const sortOrder = order === 'desc' ? -1 : 1;
    const sort = { [sortBy]: sortOrder };
    
    // Secondary sort by order field for same year
    if (sortBy === 'year') {
      sort.order = 1;
    }

    const publications = await Publication.find(query)
      .sort(sort)
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .lean();

    const count = await Publication.countDocuments(query);

    res.json({
      success: true,
      data: {
        publications,
        totalPages: Math.ceil(count / limit),
        currentPage: parseInt(page),
        total: count
      }
    });
  } catch (error) {
    console.error('Get publications error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch publications',
      error: error.message
    });
  }
};

// Get publications grouped by year
exports.getPublicationsByYear = async (req, res) => {
  try {
    const { type, featured } = req.query;
    
    const query = { isPublished: true };
    if (type) query.type = type;
    if (featured !== undefined) query.featured = featured === 'true';

    const publications = await Publication.aggregate([
      { $match: query },
      { $sort: { year: -1, order: 1 } },
      {
        $group: {
          _id: '$year',
          publications: { $push: '$$ROOT' },
          count: { $sum: 1 }
        }
      },
      { $sort: { _id: -1 } }
    ]);

    res.json({
      success: true,
      data: publications
    });
  } catch (error) {
    console.error('Get publications by year error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch publications',
      error: error.message
    });
  }
};

// Get single publication
exports.getPublicationById = async (req, res) => {
  try {
    const publication = await Publication.findById(req.params.id);

    if (!publication) {
      return res.status(404).json({
        success: false,
        message: 'Publication not found'
      });
    }

    res.json({
      success: true,
      data: publication
    });
  } catch (error) {
    console.error('Get publication error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch publication',
      error: error.message
    });
  }
};

// Create publication (admin)
exports.createPublication = async (req, res) => {
  try {
    const publication = new Publication(req.body);
    await publication.save();

    res.status(201).json({
      success: true,
      message: 'Publication created successfully',
      data: publication
    });
  } catch (error) {
    console.error('Create publication error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create publication',
      error: error.message
    });
  }
};

// Update publication (admin)
exports.updatePublication = async (req, res) => {
  try {
    const publication = await Publication.findByIdAndUpdate(
      req.params.id,
      req.body,
      { new: true, runValidators: true }
    );

    if (!publication) {
      return res.status(404).json({
        success: false,
        message: 'Publication not found'
      });
    }

    res.json({
      success: true,
      message: 'Publication updated successfully',
      data: publication
    });
  } catch (error) {
    console.error('Update publication error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update publication',
      error: error.message
    });
  }
};

// Delete publication (admin)
exports.deletePublication = async (req, res) => {
  try {
    const publication = await Publication.findByIdAndDelete(req.params.id);

    if (!publication) {
      return res.status(404).json({
        success: false,
        message: 'Publication not found'
      });
    }

    res.json({
      success: true,
      message: 'Publication deleted successfully'
    });
  } catch (error) {
    console.error('Delete publication error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete publication',
      error: error.message
    });
  }
};

// Get publication statistics
exports.getPublicationStats = async (req, res) => {
  try {
    const stats = await Publication.aggregate([
      { $match: { isPublished: true } },
      {
        $group: {
          _id: null,
          total: { $sum: 1 },
          totalCitations: { $sum: '$citations' },
          byType: {
            $push: {
              type: '$type',
              year: '$year'
            }
          }
        }
      }
    ]);

    const typeStats = await Publication.aggregate([
      { $match: { isPublished: true } },
      {
        $group: {
          _id: '$type',
          count: { $sum: 1 }
        }
      }
    ]);

    const yearStats = await Publication.aggregate([
      { $match: { isPublished: true } },
      {
        $group: {
          _id: '$year',
          count: { $sum: 1 }
        }
      },
      { $sort: { _id: -1 } }
    ]);

    res.json({
      success: true,
      data: {
        total: stats[0]?.total || 0,
        totalCitations: stats[0]?.totalCitations || 0,
        byType: typeStats,
        byYear: yearStats
      }
    });
  } catch (error) {
    console.error('Get publication stats error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch publication statistics',
      error: error.message
    });
  }
};
