const News = require('../models/News');
const { uploadToFileVila } = require('../utils/filevilaUpload');

// Get all news (with filters)
const getAllNews = async (req, res) => {
  try {
    const { page = 1, limit = 10, status, type, search = '' } = req.query;
    const skip = (page - 1) * limit;

    let query = {};
    
    // For public API, only show published
    if (status === 'published') {
      query.status = 'published';
    }
    
    // For admin, can filter by any status
    if (status && status !== 'published') {
      query.status = status;
    }

    if (type) {
      query.type = type;
    }

    if (search) {
      query.$or = [
        { title: { $regex: search, $options: 'i' } },
        { short_desc: { $regex: search, $options: 'i' } }
      ];
    }

    const news = await News.find(query)
      .populate('created_by', 'name email')
      .populate('tagged_members', 'name designation image')
      .sort({ published_at: -1, createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit));

    const total = await News.countDocuments(query);

    res.json({
      news,
      pagination: {
        current_page: parseInt(page),
        total_pages: Math.ceil(total / limit),
        total_items: total,
        has_next: page * limit < total,
        has_prev: page > 1
      }
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get single news by ID
const getNewsById = async (req, res) => {
  try {
    const { id } = req.params;
    
    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid news ID format' });
    }
    
    const news = await News.findById(id)
      .populate('created_by', 'name email')
      .populate('tagged_members', 'name designation image bio emails');

    if (!news) {
      return res.status(404).json({ error: 'News not found' });
    }

    res.json({ news });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Create news
const createNews = async (req, res) => {
  try {
    let { type, title, short_desc, content, tagged_members, status } = req.body;

    // Parse JSON strings if they're sent from FormData
    if (typeof tagged_members === 'string') {
      try {
        tagged_members = JSON.parse(tagged_members);
      } catch (e) {
        tagged_members = [];
      }
    }
    
    if (typeof content === 'string') {
      try {
        content = JSON.parse(content);
      } catch (e) {
        // Content might already be an object or valid JSON string
      }
    }

    if (!type || !title || !content) {
      return res.status(400).json({ error: 'Type, title, and content are required' });
    }

    if (title.trim().length < 3) {
      return res.status(400).json({ error: 'Title must be at least 3 characters long' });
    }

    if (title.length > 200) {
      return res.status(400).json({ error: 'Title must not exceed 200 characters' });
    }

    let thumbImageUrl = '';
    if (req.file) {
      // New file upload
      thumbImageUrl = await uploadToFileVila(req.file, 'news');
    } else if (req.body.thumb_image) {
      // URL from media library
      thumbImageUrl = req.body.thumb_image;
    }

    const news = new News({
      type,
      title,
      short_desc: short_desc || '',
      content,
      tagged_members: tagged_members || [],
      status: status || 'draft',
      created_by: req.user._id,
      thumb_image: thumbImageUrl
    });

    if (status === 'published') {
      news.published_at = new Date();
    }

    await news.save();

    const populatedNews = await News.findById(news._id)
      .populate('created_by', 'name email')
      .populate('tagged_members', 'name designation image');

    res.status(201).json({
      message: 'News created successfully',
      news: populatedNews
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update news
const updateNews = async (req, res) => {
  try {
    const { id } = req.params;
    let { type, title, short_desc, content, tagged_members, status } = req.body;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid news ID format' });
    }

    // Parse JSON strings if they're sent from FormData
    if (typeof tagged_members === 'string') {
      try {
        tagged_members = JSON.parse(tagged_members);
      } catch (e) {
        tagged_members = undefined;
      }
    }
    
    if (typeof content === 'string') {
      try {
        content = JSON.parse(content);
      } catch (e) {
        // Content might already be an object or valid JSON string
      }
    }

    const news = await News.findById(id);
    if (!news) {
      return res.status(404).json({ error: 'News not found' });
    }

    if (type) news.type = type;
    if (title) news.title = title;
    if (short_desc !== undefined) news.short_desc = short_desc;
    if (content) news.content = content;
    if (tagged_members) news.tagged_members = tagged_members;
    
    if (req.file) {
      // New file upload
      news.thumb_image = await uploadToFileVila(req.file, 'news');
    } else if (req.body.thumb_image !== undefined) {
      // URL from media library or empty string to clear
      news.thumb_image = req.body.thumb_image;
    }

    // Handle status change to published
    if (status === 'published' && news.status !== 'published') {
      news.status = 'published';
      news.published_at = new Date();
    } else if (status) {
      news.status = status;
    }

    await news.save();

    const populatedNews = await News.findById(news._id)
      .populate('created_by', 'name email')
      .populate('tagged_members', 'name designation image');

    res.json({
      message: 'News updated successfully',
      news: populatedNews
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Delete news
const deleteNews = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid news ID format' });
    }

    const news = await News.findById(id);
    if (!news) {
      return res.status(404).json({ error: 'News not found' });
    }

    await News.findByIdAndDelete(id);

    res.json({ message: 'News deleted successfully' });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Publish news
const publishNews = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid news ID format' });
    }

    const news = await News.findById(id);
    if (!news) {
      return res.status(404).json({ error: 'News not found' });
    }

    await news.publish();

    res.json({
      message: 'News published successfully',
      news
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  getAllNews,
  getNewsById,
  createNews,
  updateNews,
  deleteNews,
  publishNews
};
