const Member = require('../models/Member');
const { uploadToFileVila } = require('../utils/filevilaUpload');

// Get all members (with filters)
const getAllMembers = async (req, res) => {
  try {
    const { page = 1, limit = 50, member_type, status = 'active', search = '' } = req.query;
    const skip = (page - 1) * limit;

    let query = {};
    
    if (status) {
      query.status = status;
    }

    if (member_type) {
      query.member_type = member_type;
    }

    if (search) {
      query.$or = [
        { name: { $regex: search, $options: 'i' } },
        { designation: { $regex: search, $options: 'i' } },
        { bio: { $regex: search, $options: 'i' } }
      ];
    }

    const members = await Member.find(query)
      .sort({ member_type: 1, order: 1, name: 1 })
      .skip(skip)
      .limit(parseInt(limit));

    const total = await Member.countDocuments(query);

    res.json({
      members,
      pagination: {
        current_page: parseInt(page),
        total_pages: Math.ceil(total / limit),
        total_items: total,
        has_next: page * limit < total,
        has_prev: page > 1
      }
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get member types (unique values)
const getMemberTypes = async (req, res) => {
  try {
    const types = await Member.distinct('member_type');
    res.json({ types });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get single member by ID
const getMemberById = async (req, res) => {
  try {
    const { id } = req.params;
    
    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid member ID format' });
    }
    
    const member = await Member.findById(id);

    if (!member) {
      return res.status(404).json({ error: 'Member not found' });
    }

    res.json({ member });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Create member
const createMember = async (req, res) => {
  try {
    const { name, designation, bio, phones, emails, member_type, order, status, social_links } = req.body;

    if (!name || !designation || !member_type) {
      return res.status(400).json({ error: 'Name, designation, and member type are required' });
    }

    if (name.trim().length < 2) {
      return res.status(400).json({ error: 'Name must be at least 2 characters long' });
    }

    if (name.length > 100) {
      return res.status(400).json({ error: 'Name must not exceed 100 characters' });
    }

    // Parse JSON strings from FormData
    let parsedPhones = [];
    let parsedEmails = [];
    let parsedSocialLinks = {};

    try {
      parsedPhones = typeof phones === 'string' ? JSON.parse(phones) : (phones || []);
      parsedEmails = typeof emails === 'string' ? JSON.parse(emails) : (emails || []);
      parsedSocialLinks = typeof social_links === 'string' ? JSON.parse(social_links) : (social_links || {});
    } catch (parseError) {
      return res.status(400).json({ error: 'Invalid JSON format for phones, emails, or social_links' });
    }

    let imageUrl = '';
    if (req.file) {
      imageUrl = await uploadToFileVila(req.file, 'members');
    }

    const member = new Member({
      name,
      designation,
      bio: bio || '',
      phones: parsedPhones,
      emails: parsedEmails,
      member_type,
      order: order || 0,
      status: status || 'active',
      social_links: parsedSocialLinks,
      image: imageUrl
    });

    await member.save();

    res.status(201).json({
      message: 'Member created successfully',
      member
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update member
const updateMember = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, designation, bio, phones, emails, member_type, order, status, social_links } = req.body;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid member ID format' });
    }

    const member = await Member.findById(id);
    if (!member) {
      return res.status(404).json({ error: 'Member not found' });
    }

    // Parse JSON strings from FormData
    let parsedPhones, parsedEmails, parsedSocialLinks;
    try {
      if (phones) parsedPhones = typeof phones === 'string' ? JSON.parse(phones) : phones;
      if (emails) parsedEmails = typeof emails === 'string' ? JSON.parse(emails) : emails;
      if (social_links) parsedSocialLinks = typeof social_links === 'string' ? JSON.parse(social_links) : social_links;
    } catch (parseError) {
      return res.status(400).json({ error: 'Invalid JSON format for phones, emails, or social_links' });
    }

    if (name) member.name = name;
    if (designation) member.designation = designation;
    if (bio !== undefined) member.bio = bio;
    if (parsedPhones) member.phones = parsedPhones;
    if (parsedEmails) member.emails = parsedEmails;
    if (member_type) member.member_type = member_type;
    if (order !== undefined) member.order = order;
    if (status) member.status = status;
    if (parsedSocialLinks) member.social_links = parsedSocialLinks;
    
    if (req.file) {
      member.image = await uploadToFileVila(req.file, 'members');
    }

    await member.save();

    res.json({
      message: 'Member updated successfully',
      member
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Delete member
const deleteMember = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid member ID format' });
    }

    const member = await Member.findById(id);
    if (!member) {
      return res.status(404).json({ error: 'Member not found' });
    }

    await Member.findByIdAndDelete(id);

    res.json({ message: 'Member deleted successfully' });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  getAllMembers,
  getMemberTypes,
  getMemberById,
  createMember,
  updateMember,
  deleteMember
};
