const DynamicData = require('../models/DynamicData');

// Get all dynamic data
const getAllDynamicData = async (req, res) => {
  try {
    const { page = 1, limit = 50, search = '' } = req.query;
    const skip = (page - 1) * limit;

    let query = {};
    if (search) {
      query.key = { $regex: search, $options: 'i' };
    }

    const data = await DynamicData.find(query)
      .sort({ key: 1 })
      .skip(skip)
      .limit(parseInt(limit));

    const total = await DynamicData.countDocuments(query);

    res.json({
      data,
      pagination: {
        current_page: parseInt(page),
        total_pages: Math.ceil(total / limit),
        total_items: total,
        has_next: page * limit < total,
        has_prev: page > 1
      }
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get single dynamic data by key
const getDynamicDataByKey = async (req, res) => {
  try {
    const { key } = req.params;
    const data = await DynamicData.findOne({ key });
    
    if (!data) {
      return res.status(404).json({ error: 'Data not found' });
    }

    res.json({ data });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Create dynamic data
const createDynamicData = async (req, res) => {
  try {
    const { key, value_type, value, description } = req.body;

    if (!key || !value) {
      return res.status(400).json({ error: 'Key and value are required' });
    }

    // Validate key format (alphanumeric, underscores, hyphens)
    if (!/^[a-zA-Z0-9_-]+$/.test(key)) {
      return res.status(400).json({ error: 'Key must contain only alphanumeric characters, underscores, and hyphens' });
    }

    if (key.length > 100) {
      return res.status(400).json({ error: 'Key must not exceed 100 characters' });
    }

    // Check if key already exists
    const existing = await DynamicData.findOne({ key });
    if (existing) {
      return res.status(400).json({ error: 'Key already exists' });
    }

    const data = new DynamicData({
      key,
      value_type: value_type || 'string',
      value,
      description: description || ''
    });

    await data.save();

    res.status(201).json({
      message: 'Dynamic data created successfully',
      data
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update dynamic data
const updateDynamicData = async (req, res) => {
  try {
    const { id } = req.params;
    const { key, value_type, value, description } = req.body;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid data ID format' });
    }

    const data = await DynamicData.findById(id);
    if (!data) {
      return res.status(404).json({ error: 'Data not found' });
    }

    // If key is being changed, check if new key already exists
    if (key && key !== data.key) {
      const existing = await DynamicData.findOne({ key });
      if (existing) {
        return res.status(400).json({ error: 'Key already exists' });
      }
      data.key = key;
    }

    if (value_type) data.value_type = value_type;
    if (value !== undefined) data.value = value;
    if (description !== undefined) data.description = description;

    await data.save();

    res.json({
      message: 'Dynamic data updated successfully',
      data
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Delete dynamic data
const deleteDynamicData = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate ID format
    if (!id || !id.match(/^[0-9a-fA-F]{24}$/)) {
      return res.status(400).json({ error: 'Invalid data ID format' });
    }

    const data = await DynamicData.findById(id);
    if (!data) {
      return res.status(404).json({ error: 'Data not found' });
    }

    await DynamicData.findByIdAndDelete(id);

    res.json({ message: 'Dynamic data deleted successfully' });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  getAllDynamicData,
  getDynamicDataByKey,
  createDynamicData,
  updateDynamicData,
  deleteDynamicData
};
