const express = require('express');
const mongoose = require('mongoose');
const cors = require('cors');
const compression = require('compression');
const path = require('path');
require('dotenv').config();

// Import security middleware
const { 
  helmetConfig,
  requestSizeLimit
} = require('./middleware/security');

const app = express();

// Trust proxy
app.set('trust proxy', 1);

// Security middleware (minimal for development)
app.use(helmetConfig);
app.use(compression());

// Request size limiting
app.use(requestSizeLimit('50mb'));

// Body parsing middleware
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// CORS configuration
const corsOptions = {
  origin: process.env.ALLOWED_ORIGINS ? process.env.ALLOWED_ORIGINS.split(',') : ['http://localhost:3000', 'http://localhost:5173', 'http://localhost:5174'],
  credentials: true,
  optionsSuccessStatus: 200
};
app.use(cors(corsOptions));

// Serve static files (uploaded images)
app.use('/uploads', express.static(path.join(__dirname, '../uploads')));

// MongoDB connection
mongoose.connect(process.env.MONGODB_URI || 'mongodb://localhost:27017/cnl_kuet')
.then(() => console.log('✅ Connected to MongoDB'))
.catch((err) => console.error('❌ MongoDB connection error:', err));

// Routes (no rate limiting for development)
app.use('/api/users', require('./routes/users'));
app.use('/api/dynamic-data', require('./routes/dynamicData'));
app.use('/api/research', require('./routes/research'));
app.use('/api/members', require('./routes/members'));
app.use('/api/news', require('./routes/news'));
app.use('/api/system', require('./routes/system'));
app.use('/api/upload', require('./routes/upload'));
app.use('/api/media', require('./routes/media'));
app.use('/api/contact', require('./routes/contact'));
app.use('/api/publications', require('./routes/publications'));

// Health check endpoint (Vercel serverless optimized)
app.get('/health', async (req, res) => {
  res.json({ 
    status: 'OK',
    version: '1.0.0',
    db: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
  });
});

// Welcome endpoint
app.get('/', (req, res) => {
  res.json({
    message: 'CNL KUET API Server',
    version: '1.0.0',
    endpoints: {
      users: '/api/users',
      dynamicData: '/api/dynamic-data',
      research: '/api/research',
      members: '/api/members',
      news: '/api/news',
      system: '/api/system',
      contact: '/api/contact',
      publications: '/api/publications'
    }
  });
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('Error:', err);
  
  if (err.name === 'ValidationError') {
    return res.status(400).json({
      error: 'Validation Error',
      details: Object.values(err.errors).map(e => e.message)
    });
  }
  
  if (err.name === 'CastError') {
    return res.status(400).json({
      error: 'Invalid ID format'
    });
  }
  
  if (err.code === 11000) {
    return res.status(400).json({
      error: 'Duplicate entry',
      field: Object.keys(err.keyPattern)[0]
    });
  }

  res.status(500).json({
    error: process.env.NODE_ENV === 'production' ? 'Internal Server Error' : err.message
  });
});

// 404 handler
app.use((req, res) => {
  res.status(404).json({
    error: 'Route not found',
    method: req.method,
    url: req.originalUrl
  });
});

// Graceful shutdown handler (Vercel serverless - connections auto-managed)
process.on('SIGINT', async () => {
  console.log('\n🔄 Shutting down gracefully...');
  
  try {
    await mongoose.connection.close();
    console.log('✅ Database connection closed');
    process.exit(0);
  } catch (error) {
    console.error('❌ Error during shutdown:', error);
    process.exit(1);
  }
});

// Start server
const PORT = process.env.PORT || 3000;
const server = app.listen(PORT, () => {
  console.log(`🚀 CNL KUET API server running on port ${PORT}`);
  console.log(`📖 API Documentation: http://localhost:${PORT}/`);
  console.log(`💚 Health Check: http://localhost:${PORT}/health`);
  console.log(`🔒 Security: Complete security stack enabled`);
});

// Handle server errors
server.on('error', (error) => {
  console.error('❌ Server error:', error);
});

module.exports = app;